%module soslib

%include "enumsimple.swg"

%{
#include "soslib.h"
%}

//////////////////////////////////////////////////////////////////////
// (Java) double[] ->  (C) double*
//////////////////////////////////////////////////////////////////////
%typemap(jni)   double*  "jdoubleArray" 
%typemap(jtype) double*  "double[]" 
%typemap(jstype) double*  "double[]" 

%typemap(javain) double* "$javainput"

%typemap(in) double* (jint size) {
  int i=0;
  jboolean isCopy;
  size = jenv->GetArrayLength($input);
  $1 = new double[size];
  jdouble* jd_array = (jdouble*)jenv->GetDoubleArrayElements($input, &isCopy);

  for(i=0; i < size; i++)
  {
    $1[i] = jd_array[i];
  }

  if (isCopy == JNI_TRUE)
  {
    jenv->ReleaseDoubleArrayElements($input, jd_array, 0);
  }
}

%typemap(freearg) double* {
  delete [] $1;
}


//////////////////////////////////////////////////////////////////////
// odeSolver.h
//////////////////////////////////////////////////////////////////////

typedef struct varySettings varySettings_t;

/** Settings for batch integration with parameter variation */
struct varySettings {};

/* settings for parameter variation batch runs */
varySettings_t *VarySettings_allocate(int nrparams, int nrdesignpoints);
//
//int  VarySettings_addParameter(varySettings_t *, char *id, char *rid, double start, double end);
//
// changed in 2008-05-12 version
//
int VarySettings_addParameter(varySettings_t *, char *, char *);
//
// added in 2008-05-12 version
//
int VarySettings_addDesignPoint(varySettings_t *, double *);
int VarySettings_setValueByID(varySettings_t *, int, char *, char*, double);
//double VarySettings_getValueByID(varySettings_t *, int, char *, char*);
//

int  VarySettings_setName(varySettings_t *, int i, char *id, char *rid);
void VarySettings_setValue(varySettings_t *, int i, int j, double value);

double VarySettings_getValue(varySettings_t *, int i, int j);
const char *VarySettings_getName(varySettings_t *, int i);
const char *VarySettings_getReactionName(varySettings_t *vs, int i);

void VarySettings_dump(varySettings_t *);
void VarySettings_free();


//////////////////////////////////////////////////////////////////////
// sbmlResults.h
//////////////////////////////////////////////////////////////////////

typedef struct timeCourse timeCourse_t ;
typedef struct timeCourseArray timeCourseArray_t ;
typedef struct _SBMLResults SBMLResults_t;
//typedef struct _SBMLResultsMatrix SBMLResultsMatrix_t;
typedef struct _SBMLResultsArray SBMLResultsArray_t;

/** A simple structure containing a variable name,
   and its time courses generated by simulation routines. */
struct timeCourse {
    int timepoints;       /**< number of timepoints, including initial
                               conditions */
    char *name;           /**< variable name */
} ;

/** A simple structure containing num_val time courses */
struct timeCourseArray {
    int num_val;        /**< number of time courses  */
} ;

/** A simple structure that contains time courses - represented
    by the timeCourseArray structure - for
    SBML structures, such as species, non-constant compartments
    and parameters and reaction fluxes.
*/
struct _SBMLResults {
    timeCourse_t *time;                /**< the time points */

    /* concentration and variable parameter and compartment time series */
    timeCourseArray_t *species;       /**< time courses for all species */
    timeCourseArray_t *compartments;  /**< time courses for all non-constant
                                         compartments */
    timeCourseArray_t *parameters;    /**< time courses for all non-constant
                                         global parameters */
    timeCourseArray_t *fluxes;        /**< time courses of reaction fluxes */
} ;

/** A matrix of _SBMLResults used for batch integration with
    parameter variation via varySettings */
//struct _SBMLResultsMatrix {} ;
  //
  // added in 2008-05-12 version
  //
  struct _SBMLResultsArray
  {
//    SBMLResults_t ** results;
//    int size; /* numsteps^numparams */
  };


timeCourse_t *SBMLResults_getTime(SBMLResults_t *);  
timeCourse_t *SBMLResults_getTimeCourse(SBMLResults_t *, const char *);
int SBMLResults_getNout(SBMLResults_t *);
int SBMLResults_getNumSens(SBMLResults_t *);
const char *SBMLResults_getSensParam(SBMLResults_t *, int);
const char*TimeCourse_getName(timeCourse_t *);
int TimeCourse_getNumValues(timeCourse_t *);
double TimeCourse_getValue(timeCourse_t *, int);
double TimeCourse_getSensitivity(timeCourse_t *, int, int);
void SBMLResults_dump(SBMLResults_t *);
void SBMLResults_dumpSpecies(SBMLResults_t *);
void SBMLResults_dumpCompartments(SBMLResults_t *);
void SBMLResults_dumpParameters(SBMLResults_t *);
void SBMLResults_dumpFluxes(SBMLResults_t *);
void SBMLResults_free(SBMLResults_t *);
//void SBMLResultsMatrix_free(SBMLResultsMatrix_t *);
//SBMLResults_t *SBMLResultsMatrix_getResults(SBMLResultsMatrix_t *, int i, int j);
//
// added in 2008-05-12 version
//
void SBMLResultsArray_free(SBMLResultsArray_t *);
SBMLResults_t *SBMLResultsArray_getResults(SBMLResultsArray_t *, int i);


//////////////////////////////////////////////////////////////////////
// integratorSettings.h
//////////////////////////////////////////////////////////////////////

/* structures */
typedef struct timeSettings timeSettings_t;
typedef struct cvodeSettings cvodeSettings_t;

/** Settings for CVODE Integration

    This structure contains the basic settings for an integration run,
    like the time course, relative and absolute erros,
    integration methods etc.
    
    It also contains diverse switches for integrator behaviour, as well
    as for setting sensitivity analysis
*/
struct cvodeSettings {};

/* create and free, get and set cvodeSettings */
cvodeSettings_t *CvodeSettings_create();
cvodeSettings_t *CvodeSettings_createWithTime(double Time, int PrintStep);
cvodeSettings_t *CvodeSettings_createWith(double EndTime, int PrintStep, double Error, double RError, int Mxstep, int Method, int IterMethod, int UseJacobian, int Indefinitely, int HaltOnEvent, int SteadyState, int StoreResults, int Sensitivity, int SensMethod);
int CvodeSettings_setTime(cvodeSettings_t *, double EndTime, int PrintStep);
int CvodeSettings_setTimeStep(cvodeSettings_t *, int, double);

void CvodeSettings_setDiscreteObservation(cvodeSettings_t *);
void CvodeSettings_unsetDiscreteObservation(cvodeSettings_t *); 
int CvodeSettings_setForwAdjTimeSeriesFromData(cvodeSettings_t *, char *, int);

void CvodeSettings_setSwitches(cvodeSettings_t *, int UseJacobian, int Indefinitely, int HaltOnEvent, int SteadyState, int StoreResults, int Sensitivity, int SensMethod);
void CvodeSettings_setErrors(cvodeSettings_t *, double Error, double RError, int Mxstep);
void CvodeSettings_setError(cvodeSettings_t *, double);
void CvodeSettings_setRError(cvodeSettings_t *, double);
void CvodeSettings_setMxstep(cvodeSettings_t *, int);
void CvodeSettings_setDetectNegState(cvodeSettings_t *, int);
void CvodeSettings_setTStop(cvodeSettings_t *, int);
void CvodeSettings_setCompileFunctions(cvodeSettings_t *, int);
void CvodeSettings_setResetCvodeOnEvent(cvodeSettings_t *, int);

void CvodeSettings_setDoAdj(cvodeSettings_t *);
void CvodeSettings_unsetDoAdj(cvodeSettings_t *);
void CvodeSettings_setAdjErrors(cvodeSettings_t *, double Error, double RError);
void CvodeSettings_setAdjError(cvodeSettings_t *, double);
void CvodeSettings_setAdjRError(cvodeSettings_t *, double);
void CvodeSettings_setnSaveSteps(cvodeSettings_t *, int);
int CvodeSettings_setAdjTime(cvodeSettings_t *, double EndTime, int PrintStep);

void CvodeSettings_setMethod(cvodeSettings_t *, int, int);
void CvodeSettings_setIterMethod(cvodeSettings_t *, int);
void CvodeSettings_setMaxOrder(cvodeSettings_t *, int);
void CvodeSettings_setJacobian(cvodeSettings_t *, int);
void CvodeSettings_setIndefinitely(cvodeSettings_t *, int);
void CvodeSettings_setHaltOnEvent(cvodeSettings_t *, int);
void CvodeSettings_setHaltOnSteadyState(cvodeSettings_t *, int);
void CvodeSettings_setSteadyStateThreshold(cvodeSettings_t *, double);
void CvodeSettings_setStoreResults(cvodeSettings_t *, int);
void CvodeSettings_setSensitivity(cvodeSettings_t *, int);
void CvodeSettings_unsetSensParams(cvodeSettings_t *);
void CvodeSettings_setSensMethod(cvodeSettings_t *, int);
void CvodeSettings_dump(cvodeSettings_t *);
void CvodeSettings_free(cvodeSettings_t *);
cvodeSettings_t *CvodeSettings_clone(cvodeSettings_t *);

double CvodeSettings_getEndTime(cvodeSettings_t *);
int CvodeSettings_getPrintsteps(cvodeSettings_t *);
double CvodeSettings_getTimeStep(cvodeSettings_t *);
double CvodeSettings_getTime(cvodeSettings_t *, int);  
double CvodeSettings_getError(cvodeSettings_t *);
double CvodeSettings_getRError(cvodeSettings_t *);
int CvodeSettings_getMxstep(cvodeSettings_t *);
char *CvodeSettings_getMethod(cvodeSettings_t *);
char *CvodeSettings_getIterMethod(cvodeSettings_t *);
int CvodeSettings_getMaxOrder(cvodeSettings_t *);
int CvodeSettings_getCompileFunctions(cvodeSettings_t *);
int CvodeSettings_getResetCvodeOnEvent(cvodeSettings_t *);

int CvodeSettings_getJacobian(cvodeSettings_t *);
int CvodeSettings_getIndefinitely(cvodeSettings_t *);
int CvodeSettings_getHaltOnEvent(cvodeSettings_t *);
int CvodeSettings_getHaltOnSteadyState(cvodeSettings_t *);
double CvodeSettings_getSteadyStateThreshold(cvodeSettings_t *);
int CvodeSettings_getStoreResults(cvodeSettings_t *);
int CvodeSettings_getSensitivity(cvodeSettings_t *);
char *CvodeSettings_getSensMethod(cvodeSettings_t *);


//////////////////////////////////////////////////////////////////////
// solverError.h
//////////////////////////////////////////////////////////////////////

  /** error codes.
      codes < 0 reserved for CVODE\n
      codes 0 throu 99999 reserved for LibSBML\n
      codes 0 throu 9999 libSBML, XML layer\n
      codes 10000 throu 99999 libSBML, SBML Errors\n */
  enum errorCode
    {

      /** 10XXXX - conversion to ode model failures in odeConstruct.c */
      SOLVER_ERROR_ODE_COULD_NOT_BE_CONSTRUCTED_FOR_SPECIES  = 100000,
      SOLVER_ERROR_THE_MODEL_CONTAINS_EVENTS = 100001,
      SOLVER_ERROR_THE_MODEL_CONTAINS_ALGEBRAIC_RULES = 100002,
      SOLVER_ERROR_ODE_MODEL_COULD_NOT_BE_CONSTRUCTED = 100003,
      SOLVER_ERROR_NO_KINETIC_LAW_FOUND_FOR_REACTION = 100004,
      SOLVER_ERROR_ENTRIES_OF_THE_JACOBIAN_MATRIX_COULD_NOT_BE_CONSTRUCTED = 100005,
      SOLVER_ERROR_MODEL_NOT_SIMPLIFIED = 100006,
      SOLVER_ERROR_ENTRIES_OF_THE_PARAMETRIC_MATRIX_COULD_NOT_BE_CONSTRUCTED = 100007,
      SOLVER_ERROR_REQUESTED_PARAMETER_NOT_FOUND = 100008,
      SOLVER_ERROR_THE_MODEL_CONTAINS_PIECEWISE = 100009,

      /** 11xx30 - SBML input model failures in sbml.c */
      SOLVER_ERROR_MAKE_SURE_SCHEMA_IS_ON_PATH = 110030,
      SOLVER_ERROR_CANNOT_PARSE_MODEL = 110031,
      SOLVER_ERROR_DOCUMENTLEVEL_ONE = 1100032,
     
      /** 11XX5X - Graph Drawing Errors  in drawGraph.c */
      SOLVER_ERROR_NO_GRAPHVIZ = 110050,
      /** 11X1XX - Wrong Input Settings */
      SOLVER_ERROR_INTEGRATOR_SETTINGS = 110100,
      SOLVER_ERROR_VARY_SETTINGS = 110101,
      
      /** 12XXXX - Integration Failures in integratorInstance.c */
      SOLVER_ERROR_INTEGRATION_NOT_SUCCESSFUL = 120000,
      SOLVER_ERROR_EVENT_TRIGGER_FIRED = 120001,
      SOLVER_ERROR_CVODE_MALLOC_FAILED = 120002,
      SOLVER_ERROR_CVODE_REINIT_FAILED = 120003,

      /** 12X1XX - AST Processing Failures in processAST.c */
      /** AST evaluation in evaluateAST */
      SOLVER_ERROR_AST_UNKNOWN_NODE_TYPE = 120100,
      SOLVER_ERROR_AST_UNKNOWN_FAILURE = 120101,
      SOLVER_ERROR_AST_EVALUATION_FAILED_MISSING_VALUE = 120102,
      SOLVER_ERROR_AST_EVALUATION_FAILED_DELAY = 120103,
      SOLVER_ERROR_AST_EVALUATION_FAILED_LAMBDA = 120104,
      SOLVER_ERROR_AST_EVALUATION_FAILED_FUNCTION = 120105,
      SOLVER_ERROR_AST_EVALUATION_FAILED_FLOAT_FACTORIAL = 120106,
      SOLVER_ERROR_AST_EVALUATION_FAILED_PIECEWISE = 120107,
      SOLVER_ERROR_AST_EVALUATION_FAILED_DISCRETE_DATA = 120108,
      /** AST differentiation in differentiateAST */
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_CONSTANT = 120110,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_OPERATOR = 120111,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_LAMBDA = 120112,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_DELAY = 120114,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_FACTORIAL = 120115,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_PIECEWISE = 120117,
      SOLVER_ERROR_AST_DIFFERENTIATION_FAILED_LOGICAL_OR_RELATIONAL = 120118,
    
      /** 12X2XX - Result Writing Failures */
      SOLVER_ERROR_CVODE_RESULTS_FAILED = 120201,
      SOLVER_ERROR_SBML_RESULTS_FAILED = 120202,

      /** 12X3XX - Adjoint Solver Failures */
       
      /** 12X4XX - Objective Function Failures */      
      SOLVER_ERROR_VECTOR_V_FAILED = 120401,
      SOLVER_ERROR_OBJECTIVE_FUNCTION_FAILED = 120402,
      SOLVER_ERROR_OBJECTIVE_FUNCTION_V_VECTOR_COULD_NOT_BE_CONSTRUCTED = 120403,
      
      /** 12X5XX - Integration Messages in integratorInstance.c */
      SOLVER_MESSAGE_RERUN_WITH_OR_WO_JACOBIAN = 120500,
      SOLVER_MESSAGE_STEADYSTATE_FOUND = 120501,
      SOLVER_ERROR_UPDATE_ADJDATA = 120502, 
      SOLVER_ERROR_INITIALIZE_ADJDATA = 120503,

      /** 12X6XX - Errors and Messages in input data */
      SOLVER_MESSAGE_INTERPOLATION_OUT_OF_RANGE = 120600,
      
      /** 13XXXX - Memory Exhaustion; general */
      SOLVER_ERROR_NO_MORE_MEMORY_AVAILABLE = 130000,

      /** 13025X - Win32 Errors */
      SOLVER_ERROR_WIN32_ERROR = 130250,
      SOLVER_ERROR_WIN32_FORMAT_ERROR = 130251,

      /** 1305XX - Compilation Errors */
      SOLVER_ERROR_COMPILATION_FAILED = 130500,
      SOLVER_ERROR_CANNOT_COMPILE_JACOBIAN_NOT_COMPUTED = 130501,
      SOLVER_ERROR_AST_COMPILATION_FAILED_DATA_AST_NODE_NOT_SUPPORTED_YET = 130502,
      SOLVER_ERROR_AST_COMPILATION_FAILED_MISSING_VALUE = 130503,
      SOLVER_ERROR_AST_COMPILATION_FAILED_STRANGE_NODE_TYPE = 130504,
      SOLVER_ERROR_CANNOT_COMPILE_SENSITIVITY_NOT_COMPUTED = 130505,
      SOLVER_ERROR_GCC_FORK_FAILED = 130506,
      SOLVER_ERROR_DL_LOAD_FAILED = 130507,
      SOLVER_ERROR_DL_SYMBOL_UNDEFINED = 130508,
      SOLVER_ERROR_OPEN_FILE = 130509,
      
      /** 14XXXX - assorted API errors */
      SOLVER_ERROR_SYMBOL_IS_NOT_IN_MODEL = 140000,
      SOLVER_ERROR_ATTEMPTING_TO_COPY_VARIABLE_STATE_BETWEEN_INSTANCES_OF_DIFFERENT_MODELS = 140001,
      SOLVER_ERROR_ATTEMPTING_TO_SET_IMPOSSIBLE_INITIAL_TIME = 140002

    } ;

typedef enum errorCode errorCode_t;

  /** error types */
  enum errorType
    {
      FATAL_ERROR_TYPE = 0,
      ERROR_ERROR_TYPE = 1,
      WARNING_ERROR_TYPE = 2,
      NUMBER_OF_ERROR_TYPES = 3,
      MESSAGE_ERROR_TYPE = 4
    } ;

typedef enum errorType errorType_t;

  /* get number of stored errors  of given type */
   int SolverError_getNum(errorType_t); 

  /* get a stored error message */
   char * SolverError_getMessage(errorType_t, int errorNum);

  /* get error code */
   errorCode_t SolverError_getCode(errorType_t, int errorNum);

  /* get error code of last error stored of given type */
   errorCode_t SolverError_getLastCode(errorType_t);

  /* empty error store */
  void SolverError_clear();

#ifdef WIN32
  /* create an error from the last windows error */
  void SolverError_storeLastWin32Error(const char *context);
#endif

  /* exit the program if errors or fatals have been created. */
  void SolverError_haltOnErrors();

  /* write all errors and warnings to standard error */
  void SolverError_dump();

  /* write all errors and warnings to a string (owned by caller unless SolverError_isMemoryExhausted()) */
  char *SolverError_dumpToString();

  /* free string returned by SolverError_dumpToString */
  void SolverError_freeDumpString(char * freestr);

  /* write all errors and warnings to standard error and then empty error store*/
  void SolverError_dumpAndClearErrors();


///////////////////////////////////////////////////////////////////////////

%inline
%{
#ifdef WIN32
char* convertUTF8ToANSI(const char* src_str); 
#endif

timeCourse_t* TimeCourseArray_getTimeCourse(timeCourseArray_t* tca, int i)
{
  if ( i >=0 && i < tca->num_val)
  {
    return tca->tc[i];
  }

  return NULL;
}

SBMLResults_t* s_odeSolver2(const char* filename, cvodeSettings_t* cset)
{
  SBMLResults_t* result = NULL;

#ifdef WIN32
  char* ansi_filename = convertUTF8ToANSI(filename);
  SBMLDocument_t* d = readSBML(ansi_filename ? ansi_filename : filename);
  delete [] ansi_filename;
#else
  SBMLDocument_t* d = readSBML(filename);
#endif
  Model_t* m = SBMLDocument_getModel(d); 

  if( m != NULL )
  {
    result = Model_odeSolver(m, cset); 
  }

  SBMLDocument_free(d);

  return result;
}

SBMLResultsArray_t* s_odeSolver3(const char* filename, cvodeSettings_t* cset, varySettings_t* vset)
{
  SBMLResultsArray_t* result = NULL;

#ifdef WIN32
  char* ansi_filename = convertUTF8ToANSI(filename);
  SBMLDocument_t* d = readSBML(ansi_filename ? ansi_filename : filename);
  delete [] ansi_filename;
#else
  SBMLDocument_t* d = readSBML(filename);
#endif
  Model_t* m = SBMLDocument_getModel(d); 

  if( m != NULL )
  {
    result = Model_odeSolverBatch(m, cset, vset); 
  }

  SBMLDocument_free(d);

  return result;
}

SBMLResultsArray_t* s_odeSolver4(const char* filename, cvodeSettings_t* cset, varySettings_t* vset)
{
  SBMLResultsArray_t* result = NULL;

#ifdef WIN32
  char* ansi_filename = convertUTF8ToANSI(filename);
  SBMLDocument_t* d = readSBML(ansi_filename ? ansi_filename : filename);
  delete [] ansi_filename;
#else
  SBMLDocument_t* d = readSBML(filename);
#endif
  Model_t* m = SBMLDocument_getModel(d); 

  if( m != NULL )
  {
    result = Model_odeSolverBatch(m, cset, vset); 
  }

  SBMLDocument_free(d);

  return result;
}


#ifdef WIN32
char* convertUTF8ToANSI(const char* src_str) 
{
	int      src_slen;
	int      length;
	int      c_length;
	wchar_t* wbuf;
	char*    abuf;

	src_slen = strlen(src_str);

	//
	// UTF8 -> Unicode
	//

//	printf("Start Convert : (%d) %s\n", src_slen, src_str);

	// Check src_str length
	length = MultiByteToWideChar(CP_UTF8,0,(LPCSTR)src_str,-1,NULL,0);
	if(length == 0){
		return NULL;
	}

    wbuf = new wchar_t[length+1];

//	printf("(UTF8->Unicode) Check Length\n");

	// Convert
	c_length = MultiByteToWideChar(CP_UTF8,0,(LPCSTR)src_str,-1,wbuf,length);
	wbuf[c_length] = 0;
	if(c_length == 0) {
        delete [] wbuf;
        return NULL;
	}

//	printf("(UTF8->Unicode) Convert : %d\n",c_length);

	//
	// Unicode -> ANSI
	//

	// Check wbuf length		
        // 
        // (2008-06-04: jouraku) 
        // Added WC_NO_BEST_FIT_CHARS option to WideCharToMultiByte function to translate any Unicode 
        // characters that do not translate directly to multibyte equivalents to the default character 
        // specified by lpDefaultChar. 
        // In other words, if translating from Unicode to multibyte and back to Unicode again does not 
        // yield the same Unicodes character,  the function uses the default character.
        // (cited from a document for WideCharToMultiByte function)
        // The purpose of this change is to avoid some malicious translation which can cause
        // security problems. 
        //
	length = WideCharToMultiByte(CP_ACP,WC_NO_BEST_FIT_CHARS, (LPCWSTR)wbuf,-1,NULL,0,NULL,NULL);

//	printf("(Unicode -> ANSI) Check Length : %d\n", length);

	if(length == 0){
       delete [] wbuf;
		return NULL;
	}

	abuf = new char[length+1];

	// Convert

        // (2008-06-04: jouraku) 
        // Added WC_NO_BEST_FIT_CHARS option to WideCharToMultiByte function to translate any Unicode 
        // characters that do not translate directly to multibyte equivalents to the default character 
        // specified by lpDefaultChar. 
        // In other words, if translating from Unicode to multibyte and back to Unicode again does not 
        // yield the same Unicodes character,  the function uses the default character.
        // (cited from a document for WideCharToMultiByte function)
        // The purpose of this change is to avoid some malicious translation which can cause
        // security problems. 
        //
	c_length = WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, (LPCWSTR)wbuf,-1,abuf,length,NULL,NULL);
	abuf[c_length] = 0;

	if(c_length == 0){
       delete [] wbuf;
       delete [] abuf;
		return NULL;
	}

//	printf("(Unicode -> ANSI) Convert : (%d) %s\n",c_length,abuf);

   delete [] wbuf;

	return abuf;

}
#endif

%}

